"""
    Simulations for Pure SU(2) gauge theory
        - Gradient flow
"""

using CUDA, Logging, StructArrays, Random, TimerOutputs, ADerrors, Printf, BDIO


CUDA.allowscalar(false)
import Pkg
Pkg.activate("/home/gtelo/PhD/SU2proj/code/runs_su2higgs/latticegpu-su2-higgs")
using LatticeGPU

# lp = SpaceParm{4}((16,16,16,16), (4,4,4,4))
lp = SpaceParm{4}((8,8,8,8), (4,4,4,4))
beta = 2.5
gp = GaugeParm(beta, 1.0, (0.0,0.0), 2)

# NSC = tryparse(Int64, ARGS[1])
println("Space  Parameters: \n", lp)
println("Gauge  Parameters: \n", gp)
GRP  = SU2
ALG  = SU2alg
SCL  = SU2fund
PREC = Float64
println("Precision:         ", PREC)

println("Allocating YM workspace")
ymws = YMworkspace(GRP, PREC, lp)

# Seed RNG
println("Seeding CURAND...")
Random.seed!(CURAND.default_rng(), 1234)
Random.seed!(1234)

# Main program
# Gauge fields U_μ=1
println("Allocating gauge field")
U = vector_field(GRP{PREC}, lp)
fill!(U, one(GRP{PREC}))

dt  = 0.05
nsteps  = 100

nth    = 20 #thermalization length
niter  = 500 # MC length
# Observables
pl     = Vector{Float64}(undef, niter+nth)
dh     = Vector{Float64}(undef, niter+nth)
acc    = Vector{Bool}(undef, niter+nth)

# FLOW PARAMETERS
Uflw = vector_field(GRP{PREC}, lp) # field used for the flow
flw_steps = 50 # GF total flow calls per MC element
flw_s     = 10 # steps of integration for each call
flw_int   = 2 # MC gap between each flow call
flw_dt    = 0.01 # fixed step size flow

flw_iter  = convert(Int64, niter/flw_int) #total number of flow measurements

gfinfo = string("_GF_stps",flw_steps,"_s",flw_s,"_int",flw_int,"_dt",flw_dt)
# flow
flwtime = Vector{Float64}(undef, flw_steps)
for f in 1:flw_steps
    flwtime[f] = f*flw_s*flw_dt
end
E       = Array{Float64, 2}(undef, flw_iter, flw_steps)
dE      = Array{Float64, 2}(undef, flw_iter, flw_steps)
Ecl     = Array{Float64, 2}(undef, flw_iter, flw_steps)
dEcl    = Array{Float64, 2}(undef, flw_iter, flw_steps)

# Save observables in BDIO file
# Uinfo 1:  Simulation parameters Int64
# Uinfo 2:  Simulations parameters Float64
# Uinfo 3:  Measurements   plaquette
# Uinfo 4:  Flow Measurements time
# Uinfo 5:  Flow Measurements E - plaquette
# Uinfo 6:  Flow Measurements dE/dt - plaquette
# Uinfo 7:  hash
# Uinfo 8:  Flow Measurements E - clover
# Uinfo 9:  Flow Measurements dE/dt - clover
# Uinfo 12: Measurement+Thermalization    dh

# Lattice dimensions
global dm = ""
for i in 1:lp.ndim-1
    global dm *= string(lp.iL[i])*"x"
end
dm *= string(lp.iL[end])
filename = string("pureYM/pureYM_",dm,"_beta",beta,"_niter", niter,"_eps",dt,"_nsteps",nsteps,gfinfo,".bdio")
fb = BDIO_open(filename, "d",
               "Pure Yang-Mills Simulations")
# Simulation param
iv = [nth, niter,flw_s,flw_steps,flw_int,flw_iter]
BDIO_start_record!(fb, BDIO_BIN_INT64LE, 1, true)
BDIO_write!(fb, iv)
BDIO_write_hash!(fb)

BDIO_start_record!(fb, BDIO_BIN_F64LE, 2, true)
s_parms = [beta, flw_dt]
BDIO_write!(fb, s_parms)
BDIO_write_hash!(fb)

#################################### SIMULATION ###################################
for sim in 1:1
    k = 0
    for i in 1:1
        HMC!(U, dt,nsteps,lp, gp, ymws; noacc=true)
        # Thermalization
        for j in 1:nth
            k += 1
            dh[k], acc[k] = HMC!(U, dt,nsteps,lp, gp, ymws)
            pl[k]  = plaquette(U,lp, gp, ymws)

            @printf("  THM %d/%d :   %s   %6.2e    %20.12e\n",
                    j, nth, acc[k] ? "true " : "false", dh[k],pl[k])
        end
        println(" ")

        # MC chain
        kflw=0
        jflw=0
        for j in 1:niter
            k += 1
            dh[k], acc[k] = HMC!(U, dt,nsteps,lp, gp, ymws)
            pl[k]  = plaquette(U,lp, gp, ymws)

            @printf("  MSM %d/%d:   %s   %6.2e    %20.12e\n",
                    j, niter, acc[k] ? "true " : "false", dh[k],
                    pl[k])

            # flow every 'flw_int'
            kflw += 1
            if (kflw == flw_int)
                print("\n\t## START Flow:")
                jflw+=1
                Uflw .= U
                for f in 1:flw_steps
                    #step flows
                    wfl_rk3(Uflw,flw_s,flw_dt,lp,ymws)
                    # wfl_euler(Uflw,flw_s,flw_dt,lp,ymws)
                    # E(f)
                    eoft = Eoft_plaq(Uflw, gp, lp, ymws)
                    E[jflw,f] = eoft
                    deoft = dEoft_plaq(Uflw, gp, lp, ymws)
                    dE[jflw,f] = deoft
                    eoft = Eoft_clover(Uflw, gp, lp, ymws)
                    Ecl[jflw,f] = eoft
                    deoft = dEoft_clover(Uflw, gp, lp, ymws)
                    dEcl[jflw,f] = deoft

                end
                kflw = 0
                println("\n\t## END flow measurements")
            end
        end
    end

    ################################### SAVE TO FILE ##################################
    # Write to BDIO
    # Plaquette
    BDIO_start_record!(fb, BDIO_BIN_F64LE, 3, true)
    BDIO_write!(fb, pl[begin:end])
    BDIO_write_hash!(fb)

    # flow time
    BDIO_start_record!(fb, BDIO_BIN_F64LE, 4, true)
    BDIO_write!(fb, flwtime)
    BDIO_write_hash!(fb)
    # E
    BDIO_start_record!(fb, BDIO_BIN_F64LE, 5, true)
    BDIO_write!(fb, E)
    BDIO_write_hash!(fb)
    # dE
    BDIO_start_record!(fb, BDIO_BIN_F64LE, 6, true)
    BDIO_write!(fb, dE)
    BDIO_write_hash!(fb)

    # E
    BDIO_start_record!(fb, BDIO_BIN_F64LE, 8, true)
    BDIO_write!(fb, Ecl)
    BDIO_write_hash!(fb)
    # dE
    BDIO_start_record!(fb, BDIO_BIN_F64LE, 9, true)
    BDIO_write!(fb, dEcl)
    BDIO_write_hash!(fb)



    #Energy conservation
    BDIO_start_record!(fb, BDIO_BIN_F64LE, 12, true)
    BDIO_write!(fb, dh[begin:end])
    BDIO_write_hash!(fb)

end

println("\n\n")

println("## Timming results")
print_timer(linechars = :ascii)

BDIO_close!(fb)

println("## END")
# END
